const nodemailer = require('nodemailer');
const { getTemplate } = require('./emailTemplates');
const winston = require('winston');
require('dotenv').config();

// Configure Winston logger
const logger = winston.createLogger({
    level: 'info',
    format: winston.format.combine(
        winston.format.timestamp(),
        winston.format.json()
    ),
    transports: [
        new winston.transports.File({ filename: 'logs/email-error.log', level: 'error' }),
        new winston.transports.File({ filename: 'logs/email.log' }),
        new winston.transports.Console({
            format: winston.format.combine(
                winston.format.colorize(),
                winston.format.simple()
            )
        })
    ]
});

// Initialize nodemailer transporter
const transporter = nodemailer.createTransport({
    host: process.env.EMAIL_HOST,
    port: parseInt(process.env.EMAIL_PORT),
    secure: process.env.EMAIL_SECURE === 'true',
    auth: {
        user: process.env.EMAIL_USER,
        pass: process.env.EMAIL_PASS
    },
    pool: true, // Enable connection pooling for better performance
    maxConnections: 5,
    maxMessages: 100,
    rateLimit: 10, // Limit to 10 emails per second
    logger: true, // Enable nodemailer logging
    debug: process.env.NODE_ENV !== 'production' // Enable debug in non-production
});

// Verify transporter configuration on startup
transporter.verify((error, success) => {
    if (error) {
        logger.error('❌ Email transporter verification failed:', error.message);
    } else {
        logger.info('✅ Email transporter is ready');
    }
});

exports.sendEmailWithTemplate = async ({ to, templateName, data, attachments = [], priority = 'normal' }) => {
    try {
        // Validate input parameters
        if (!to || !templateName) {
            throw new Error('Missing required parameters: to and templateName are required');
        }

        // Get email template
        const { subject, html } = getTemplate(templateName, data);
        console.log('🧪 Email template data:', data);

        // Configure mail options
        const mailOptions = {
            from: `"${process.env.EMAIL_SENDER_NAME}" <${process.env.EMAIL_FROM}>`,
            to: Array.isArray(to) ? to.join(', ') : to,
            subject,
            html,
            attachments,
            priority,
            headers: {
                'X-EMTAF-Application': 'Bursary-Portal',
                'X-Timestamp': new Date().toISOString()
            }
        };

        // Send email
        const info = await transporter.sendMail(mailOptions);
        logger.info(`📧 Email sent to ${to}: ${info.messageId}`, {
            template: templateName,
            response: info.response
        });

        return {
            success: true,
            messageId: info.messageId,
            response: info.response
        };
    } catch (error) {
        logger.error('❌ Email sending failed:', {
            error: error.message,
            to,
            template: templateName
        });
        return {
            success: false,
            error: error.message
        };
    }
};

// Function to queue emails for bulk sending
exports.sendBulkEmails = async (emailJobs) => {
    const results = [];
    for (const job of emailJobs) {
        const result = await exports.sendEmailWithTemplate(job);
        results.push(result);
    }
    return results;
};

// Handle transporter errors
transporter.on('error', (error) => {
    logger.error('Transporter error:', error.message);
});

// Graceful shutdown
process.on('SIGTERM', () => {
    logger.info('Closing email transporter...');
    transporter.close();
});