const app = require('./app');
const sequelize = require('./config/db');
const winston = require('winston');
require('dotenv').config();

// Configure Winston logger
const logger = winston.createLogger({
    level: 'info',
    format: winston.format.combine(
        winston.format.timestamp(),
        winston.format.json()
    ),
    transports: [
        new winston.transports.File({ filename: 'logs/error.log', level: 'error' }),
        new winston.transports.File({ filename: 'logs/combined.log' }),
        new winston.transports.Console({
            format: winston.format.combine(
                winston.format.colorize(),
                winston.format.simple()
            )
        })
    ]
});

const PORT = process.env.PORT || 3001;

// Graceful shutdown handler
const gracefulShutdown = async (signal) => {
    logger.info(`${signal} received. Shutting down gracefully...`);
    try {
        await sequelize.close();
        logger.info('Database connection closed.');
        process.exit(0);
    } catch (err) {
        logger.error('Error during shutdown:', err);
        process.exit(1);
    }
};

// Handle process termination signals
process.on('SIGINT', () => gracefulShutdown('SIGINT'));
process.on('SIGTERM', () => gracefulShutdown('SIGTERM'));

// Main application bootstrap
(async () => {
    try {
        // Test database connection
        await sequelize.authenticate();
        logger.info('✅ Database connection established successfully.');

        // Sync database schema
        // await sequelize.sync({ alter: process.env.NODE_ENV !== 'production' });
        await sequelize.sync({
            alter: true,
            logging: (msg) => console.log('🔍 Sequelize:', msg) // Log SQL queries
        });
        logger.info('✅ Database synchronized successfully with logging.');

        // Start server
        const server = app.listen(PORT, () => {
            logger.info(`🚀 Server running on port ${PORT} in ${process.env.NODE_ENV || 'development'} mode`);
        });

        // Handle server errors
        server.on('error', (err) => {
            logger.error('Server error:', err);
        });

        // Set up health check endpoint
        app.get('/health', (req, res) => {
            res.status(200).json({
                status: 'OK',
                uptime: process.uptime(),
                timestamp: new Date().toISOString()
            });
        });

    } catch (err) {
        logger.error('❌ Failed to start server:', err);
        process.exit(1);
    }
})();