const { DataTypes } = require('sequelize');
const bcrypt = require('bcryptjs');
const sequelize = require('../config/db');

const User = sequelize.define('User', {
  id: {
    type: DataTypes.UUID,
    defaultValue: DataTypes.UUIDV4,
    primaryKey: true,
  },
  fullName: {
    type: DataTypes.STRING,
    allowNull: false,
  },
  email: {
    type: DataTypes.STRING,
    unique: true,
    allowNull: false,
    validate: { isEmail: true },
  },
  password: {
    type: DataTypes.STRING,
    allowNull: false,
  },
  role: {
    type: DataTypes.ENUM('student', 'admin', 'systemAdmin'),
    defaultValue: 'student',
  },
  address: DataTypes.STRING,
  phone: DataTypes.STRING,
  gender: {
    type: DataTypes.ENUM('male', 'female', 'other'),
    allowNull: true,
  },
  race: DataTypes.STRING,
  idNumber: DataTypes.STRING,
  dateOfBirth: DataTypes.DATEONLY,
  institution: DataTypes.STRING,
  qualification: DataTypes.STRING,
  guardianName: DataTypes.STRING,
  guardianContact: DataTypes.STRING,

  // Security & Recovery
  resetToken: DataTypes.STRING,
  resetTokenExpiry: DataTypes.DATE,
  emailVerified: {
    type: DataTypes.BOOLEAN,
    defaultValue: false,
  },
  twoFactorEnabled: {
    type: DataTypes.BOOLEAN,
    defaultValue: false,
  },

  // System Metadata
  disabilityStatus: {
    type: DataTypes.BOOLEAN,
    defaultValue: false,
  },
  lastLoginAt: DataTypes.DATE,
  termsAcceptedAt: DataTypes.DATE,
  createdBy: DataTypes.UUID,

}, {
  timestamps: true,
  hooks: {
    beforeCreate: async (user) => {
      if (user.password) {
        user.password = await bcrypt.hash(user.password, 10);
      }
    },
    beforeUpdate: async (user) => {
      if (user.changed('password')) {
        user.password = await bcrypt.hash(user.password, 10);
      }
    }
  },
  indexes: [
    { fields: ['email'] },
    { fields: ['resetToken'] },
    { fields: ['idNumber'] },
  ]
});

// Method to compare plain password
User.prototype.comparePassword = async function (plainPassword) {
  return await bcrypt.compare(plainPassword, this.password);
};

module.exports = User;
