const { createControllerErrorHandler, throwAPIError } = require('../../utils/errorHandler');
const Query = require('../../models/Query');
const mailer = require('../../config/mailer');

const handleError = createControllerErrorHandler('AdminQueryController');

const getQueries = async (req, res, next) => {
    try {
        const { status, page = 1, limit = 10 } = req.query;
        const query = status ? { status } : {};

        const queries = await Query.find(query)
            .populate('userId', 'firstName lastName email')
            .skip((page - 1) * limit)
            .limit(parseInt(limit))
            .sort({ createdAt: -1 });

        const total = await Query.countDocuments(query);

        res.status(200).json({
            status: 'success',
            data: { queries, total, page: parseInt(page), limit: parseInt(limit) },
        });
    } catch (error) {
        next(handleError(error));
    }
};

const respondToQuery = async (req, res, next) => {
    try {
        const { queryId, response } = req.body;
        const query = await Query.findById(queryId).populate('userId');
        if (!query) throwAPIError(404, 'Query not found');

        query.response = response;
        query.status = 'resolved';
        query.responseDate = Date.now();
        await query.save();

        await mailer.sendQueryResponse({
            to: query.userId.email,
            userName: `${query.userId.firstName} ${query.userId.lastName}`,
            queryId: query._id,
            response,
        });

        res.status(200).json({
            status: 'success',
            data: { query },
        });
    } catch (error) {
        next(handleError(error));
    }
};

module.exports = { getQueries, respondToQuery };