const { createControllerErrorHandler, throwAPIError } = require('../../utils/errorHandler');
const Payment = require('../../models/Payment');
const Application = require('../../models/Application');

const handleError = createControllerErrorHandler('AdminPaymentController');

const createPayment = async (req, res, next) => {
    try {
        const { applicationId, institution, amount, invoiceFilePath } = req.body;
        const application = await Application.findById(applicationId).populate('userId');
        if (!application) throwAPIError(404, 'Application not found');
        if (application.status !== 'approved') throwAPIError(400, 'Application must be approved');

        const payment = await Payment.create({
            applicationId,
            userId: application.userId._id,
            institution,
            amount,
            invoiceFilePath,
            status: 'pending',
        });

        res.status(201).json({
            status: 'success',
            data: { payment },
        });
    } catch (error) {
        next(handleError(error));
    }
};

const updatePaymentStatus = async (req, res, next) => {
    try {
        const { paymentId, status, proofOfPaymentFilePath } = req.body;
        const validStatuses = ['requested', 'authorized', 'paid', 'failed'];
        if (!validStatuses.includes(status)) throwAPIError(400, 'Invalid payment status');

        const payment = await Payment.findById(paymentId);
        if (!payment) throwAPIError(404, 'Payment not found');

        payment.status = status;
        if (status === 'paid') payment.paymentDate = Date.now();
        if (proofOfPaymentFilePath) payment.proofOfPaymentFilePath = proofOfPaymentFilePath;
        await payment.save();

        res.status(200).json({
            status: 'success',
            data: { payment },
        });
    } catch (error) {
        next(handleError(error));
    }
};

module.exports = { createPayment, updatePaymentStatus };