const fs = require('fs');
const Document = require('../models/Document');
const Application = require('../models/Application');
const path = require('path');

exports.uploadDocument = async (req, res) => {
    try {
        const { applicationId, type } = req.body;
        const userId = req.user.id;

        // Check if app belongs to user
        const app = await Application.findOne({
            where: { id: applicationId, userId },
        });
        if (!app) return res.status(404).json({ message: 'Application not found' });

        const file = req.file;
        if (!file) return res.status(400).json({ message: 'No file uploaded' });

        const doc = await Document.create({
            applicationId,
            type,
            filePath: file.path,
            originalName: file.originalname,
        });

        res.status(201).json({
            message: 'Document uploaded successfully',
            document: doc,
        });
    } catch (err) {
        console.error('❌ Upload error:', err);
        res.status(500).json({ message: 'Server error' });
    }
};

exports.getDocumentsByApplication = async (req, res) => {
    try {
        const { applicationId } = req.params;
        const userId = req.user.id;

        const app = await Application.findOne({
            where: { id: applicationId, userId },
        });

        if (!app) return res.status(404).json({ message: 'Application not found' });

        const documents = await Document.findAll({ where: { applicationId } });
        res.json({ documents });
    } catch (err) {
        console.error('❌ View documents error:', err);
        res.status(500).json({ message: 'Server error' });
    }
};

exports.deleteDocument = async (req, res) => {
    try {
        const { documentId } = req.params;
        const userId = req.user.id;

        const doc = await Document.findByPk(documentId, {
            include: { model: Application },
        });

        if (!doc || doc.Application.userId !== userId) {
            return res.status(404).json({ message: 'Document not found or unauthorized' });
        }

        // Delete file from disk
        fs.unlink(doc.filePath, async (err) => {
            if (err) console.warn('⚠️ File missing on disk:', err.message);
        });

        await doc.destroy();
        res.json({ message: 'Document deleted successfully' });
    } catch (err) {
        console.error('❌ Delete document error:', err);
        res.status(500).json({ message: 'Server error' });
    }
};