const winston = require('winston');
const chalk = require('chalk');
const User = require('../../models/User');
const Application = require('../../models/Application');
const BursaryContract = require('../../models/BursaryContract');
const pageTitles = require('../../config/pageTitles');

const getDashboard = async (req, res) => {
  try {
    if (!req.user || !['admin', 'systemAdmin'].includes(req.user.role)) {
      return res.status(403).render('errors/403', {
        title: 'Unauthorized',
        message: 'You do not have permission to access this page.',
        user: req.user || null,
      });
    }

    // MongoDB stats
    const totalStudents = await User.countDocuments({ role: 'user' });
    const totalApplications = await Application.countDocuments();
    const approvedBursaries = await BursaryContract.countDocuments({ status: 'approved' });

    const dashboardData = {
      user: {
        email: req.user.email,
        role: req.user.role,
      },
      stats: {
        totalStudents,
        totalApplications,
        approvedBursaries,
      },
    };

    // Logging
    winston.info('Dashboard accessed', {
      context: 'Dashboard',
      email: req.user.email,
      role: req.user.role,
      timestamp: new Date().toISOString(),
    });

    if (process.env.NODE_ENV !== 'production') {
      console.log(
        chalk.green.bold('✅ Dashboard accessed'),
        '\n',
        chalk.cyan(`Email: ${req.user.email}`),
        '\n',
        chalk.cyan(`Role: ${req.user.role}`),
        '\n',
        chalk.gray(`Time: ${new Date().toISOString()}`)
      );
    }

    // Render EJS dashboard
    res.render('admin/dashboard/index', {
      title: pageTitles.dashboard,
      dashboardData,
      user: req.user,
    });
  } catch (err) {
    winston.error('Dashboard access error', {
      context: 'Dashboard',
      email: req.user?.email || 'unknown',
      stack: err.stack,
      timestamp: new Date().toISOString(),
    });

    if (process.env.NODE_ENV !== 'production') {
      console.error(
        chalk.red.bold('🚨 Dashboard access error'),
        '\n',
        chalk.red(`Message: ${err.message}`),
        '\n',
        chalk.gray(`Email: ${req.user?.email || 'unknown'}`),
        '\n',
        chalk.gray(`Stack: ${err.stack}`),
        '\n',
        chalk.gray(`Time: ${new Date().toISOString()}`)
      );
    }

    res.status(500).render('errors/500', {
      title: 'Dashboard Error',
      message: 'An error occurred while accessing the dashboard.',
      user: req.user || null,
    });
  }
};

module.exports = { /* The `getDashboard` function is an asynchronous function that handles the logic
for rendering the dashboard page for admin or system admin users. Here is a
breakdown of what the function does: */
getDashboard };
