const { Bursary } = require('../models/Bursary'); // Direct if self-contained
const { v4: uuidv4 } = require('uuid');

/**
 * @desc Create a new bursary (Admin)
 */
exports.createBursary = async (req, res) => {
  try {
    const {
      title,
      description,
      academicYear,
      institution,
      qualification,
      maxAmount,
      applicationDeadline,
    } = req.body;

    const createdBy = req.user.id;

    const bursary = await Bursary.create({
      id: uuidv4(),
      title,
      description,
      academicYear,
      institution,
      qualification,
      maxAmount,
      applicationDeadline,
      createdBy,
    });

    res.status(201).json({ message: 'Bursary created', bursary });
  } catch (error) {
    console.error('❌ Error creating bursary:', error);
    res.status(500).json({ message: 'Internal Server Error' });
  }
};

/**
 * @desc Get all bursaries (Admin view)
 */
exports.getAllBursaries = async (req, res) => {
  try {
    const bursaries = await Bursary.findAll({
      order: [['createdAt', 'DESC']],
    });
    res.status(200).json({ bursaries });
  } catch (error) {
    console.error('❌ Error fetching bursaries:', error);
    res.status(500).json({ message: 'Internal Server Error' });
  }
};

/**
 * @desc Get bursary by ID
 */
exports.getBursaryById = async (req, res) => {
  try {
    const bursary = await Bursary.findByPk(req.params.id);
    if (!bursary) {
      return res.status(404).json({ message: 'Bursary not found' });
    }
    res.status(200).json({ bursary });
  } catch (error) {
    console.error('❌ Error getting bursary:', error);
    res.status(500).json({ message: 'Internal Server Error' });
  }
};

/**
 * @desc Update bursary by ID
 */
exports.updateBursary = async (req, res) => {
  try {
    const bursary = await Bursary.findByPk(req.params.id);
    if (!bursary) {
      return res.status(404).json({ message: 'Bursary not found' });
    }

    await bursary.update(req.body);

    res.status(200).json({ message: 'Bursary updated', bursary });
  } catch (error) {
    console.error('❌ Error updating bursary:', error);
    res.status(500).json({ message: 'Internal Server Error' });
  }
};

/**
 * @desc Delete bursary
 */
exports.deleteBursary = async (req, res) => {
  try {
    const bursary = await Bursary.findByPk(req.params.id);
    if (!bursary) {
      return res.status(404).json({ message: 'Bursary not found' });
    }

    await bursary.destroy();
    res.status(200).json({ message: 'Bursary deleted' });
  } catch (error) {
    console.error('❌ Error deleting bursary:', error);
    res.status(500).json({ message: 'Internal Server Error' });
  }
};
