const Application = require('../models/Application');
const User = require('../models/User');
const Bursary = require('../models/Bursary');
const { sendEmailWithTemplate } = require('../utils/emailService');

const generateRefNumber = () => {
    const year = new Date().getFullYear();
    const rand = Math.floor(100000 + Math.random() * 900000);
    return `EMTAF-${year}-${rand}`;
};

exports.applyForBursary = async (req, res) => {
    try {
        const {
            bursaryId,
            qualification,
            institution,
            tuitionFee,
            bookFee,
            accommodationFee,
            motivation
        } = req.body;

        // Optional: Check if bursary exists and is still open
        const bursary = await Bursary.findByPk(bursaryId);
        if (!bursary) {
            return res.status(404).json({ message: 'Bursary not found' });
        }

        if (new Date(bursary.applicationDeadline) < new Date()) {
            return res.status(400).json({ message: 'This bursary application deadline has passed' });
        }

        const userId = req.user.id;

        const totalRequested =
            parseFloat(tuitionFee) +
            parseFloat(bookFee) +
            parseFloat(accommodationFee);

        const refNumber = generateRefNumber();

        const application = await Application.create({
            userId,
            bursaryId,
            refNumber,
            qualification,
            institution,
            tuitionFee,
            bookFee,
            accommodationFee,
            totalRequested,
            motivation,
            yearOfStudy,
            academicYear,
            applicationSource: 'online',
        });

        // ✅ Fetch user (req.user should contain fullName + email)
        const user = req.user;

        // ✅ Send email notification
        await sendEmailWithTemplate({
            to: user.email,
            templateName: 'applicationSubmitted',
            data: {
                user,
                application,
            },
        });

        res.status(201).json({
            message: 'Application submitted successfully',
            application,
        });
    } catch (error) {
        console.error('❌ Apply for bursary error:', error);
        res.status(500).json({ message: 'Internal Server Error' });
    }
};

exports.getMyApplications = async (req, res) => {
    try {
        const userId = req.user.id;

        const applications = await Application.findAll({
            where: { userId },
            order: [['createdAt', 'DESC']],
        });

        res.status(200).json({ applications });
    } catch (error) {
        console.error('❌ Error fetching applications:', error);
        res.status(500).json({ message: 'Internal Server Error' });
    }
};

exports.getApplicationByRef = async (req, res) => {
    try {
        const userId = req.user.id;
        const { refNumber } = req.params;

        const application = await Application.findOne({
            where: { userId, refNumber },
        });

        if (!application) {
            return res.status(404).json({ message: 'Application not found' });
        }

        res.status(200).json({ application });
    } catch (error) {
        console.error('❌ Error fetching application by ref:', error);
        res.status(500).json({ message: 'Internal Server Error' });
    }
};
